function [agent,GlobalFrames,Global_Entropy]=CognitiveQuantum(Npop,Niter,RandomPersonality,Nclusters,DecisionModel,InterventionType,seed,ImageFrames)
% matlab code used in Kirsty & Boschetti, 2013, "ATTITUDES, IDEOLOGIES AND SELF-ORGANISATION:
%                                               INFORMATION LOAD MINIMISATION IN MULTI-AGENT DECISION MAKING"
% Input
% Npop=number of agents
% Niter=number of iterations
% RandomPersonality = (0,1) (1= random conformity and consistency
%                           (0= conformity and consistency =.5 for each agent) 
% Nclusters= number of global frames
% DecisionModel= ('Weighted','Ideology2',) how to interpret the conformity and consistency weigths
%                                          (see section 5.1. 'Implementation' in the above mentioned paper)
% InterventionType=('PartyStatement','NewIssue','None') type of intervetion in the system; 
%                                          (see sections 5.4 and 6 in the above mentioned paper)
% seed= random seed (0=rendom,otherwise use the random seed to replicate a run)
% ImageFrames= at what iteration time you want to take a snapshot
% 
% Examples:
% 
% to reproduce Figure 9 in the above mentioned paper
% CognitiveQuantum(100,100,1,2,'Ideology2','PartyStatement',210843,[51 60 70 90 100]);
% to reproduce Figure 8 in the above mentioned paper
% CognitiveQuantum(100,100,1,2,'Ideology2','NewIssue',210843,[51 60 70 90 100]);
% to reproduce Figure 5 in the above mentioned paper
% CognitiveQuantum(100,100,1,2,'Ideology2','None',210843,[1 10 20 30 40 50 65 80 100]);
%--------------------------------------------------------------------------
%- Initialisation ---------------------------------------------------------
%how to interpret the conformity and consistency weigths
FigurePos=[900 300 500 500];FontSize=16;
ClusterMethod='Kmeans';
MakeMovie=0;debug=0;ShowStatus=10;%how often it should print the frame position
span=180;%size of the circle in which the angels are projected
% things we do not need to touch
ent=zeros(1,Npop);Global_Entropy=zeros(1,Niter);act=zeros(Niter,Npop);
%--------------------------------------------------------------------------
%--figure stuff----------------------------------------------------------
%--------------------------------------------------------------------------
hFig=figure(1);clf;
set(hFig,'Units','pixels','Position',FigurePos);
axis tight;set(gca,'NextPlot','replaceChildren');
% -----------------------------------------
% Random stuff
% ------------------------------------------
[agent,GlobalFrames,randomNumbers,seed]=RandomStuff(Npop,Niter,seed,RandomPersonality,Nclusters,span);
ImageName=strcat(InterventionType,'_',num2str(seed),'_');
%--------------------------------------------------------------------------
%--Intervention stuff------------------------------------------------------
%--------------------------------------------------------------------------
InterventionTime=1/2;%when do we intervene in the system, as multiple of Niter
InterventionTime=InterventionTime*Niter+1;%when intervention begins
InterventionLength=20;%number of iterations the perturbation is imposed for 
InterventionPeriod=[InterventionTime:InterventionTime+InterventionLength];
InterventionStrenght=40;%how many degress is the frame moved
KickedFrame=2;%Which Frame is kicked
%--------------------------------------------------------------------------
%--Iteration Loop----------------------------------------------------------
%--------------------------------------------------------------------------
frameCounter=0;
for t=1:Niter   
    probability=randomNumbers(t,:);%probability used to decide what actions an agent will take
    %--Clustering----------------------------------------------------------
    [GlobalFrames,PartyMembers]=ChooseClusters(span,Npop,Nclusters,agent,GlobalFrames);
    %-Check if anything needs to be done at thsi interation--------------------  
    switch InterventionType
        case 'None' %no intervention   
        case 'NewIssue' %resets the frames as a result of a new question
            if isequal(t,InterventionTime),agent.frame=rand(1,Npop)*span;end
        case 'PartyStatement'%party has a media campaign
            if isequal(t,InterventionTime), 
                Newframe=GlobalFrames(KickedFrame)+InterventionStrenght;
            end
            if ismember(t,InterventionPeriod), 
                GlobalFrames(KickedFrame)=Newframe;
            end
        otherwise
        error('Intervention Type missing')
    end
%--------------------------------------------------------------------------
%--Agent Loop----------------------------------------------------------
%--------------------------------------------------------------------------
    for i=1:Npop        
%       decide---------------
        [Update_Global,Update_Local,action,ent(i),chosenFrame]=DecisionMaking(t,agent.frame(i),agent.state(i),agent.comformity(i),agent.consistency(i),DecisionModel,probability(i),GlobalFrames);
        act(t,i)=action;
%       update frames--------------
        [agent.state(i),agent.frame(i)]=Act(action,agent.state(i),agent.frame(i),GlobalFrames(chosenFrame),Update_Local,Update_Global,span);
    end
%------------------------------------------------------------------------
    Global_Entropy(t)=sum(ent);%entropy bookkeeping for plotting
    PlotPolar2(t,Npop,Nclusters,agent,GlobalFrames,FontSize,ImageFrames,ImageName,hFig);
    if MakeMovie,frameCounter=frameCounter+1;F(frameCounter) = getframe(gcf);end
    if ~mod(t,ShowStatus),WriteFrames(GlobalFrames,PartyMembers,t);end
end

h2=figure(2);clf;
set(h2,'Units','pixels','Position',FigurePos);
a1 = axes;
set(a1,'OuterPosition',[0 .25 1 .55])
plot(Global_Entropy,'k-','LineWidth',2);
filename=strcat(ImageName,'_Ent.png');
title('Entropy evolution','fontsize',FontSize,'fontweight','bold')
set(a1,'Fontweight','b');%for ticks
print(h2,'-dpng',filename)

%     axis square
% plot and write outcome
act=act(:);h=hist(act,[1:4]);fprintf('final action distribution =%g,%g,%g,%g\n',h)
WriteFrames(GlobalFrames,PartyMembers,t)
fprintf('final entropy =%g\n',Global_Entropy(end))
fprintf('seed used=%g\n',seed);
if MakeMovie,movie2avi(F, 'myMovie.avi', 'compression', 'None','fps',5,'quality',100);end
%---------------------------------------------------------------------------
%---------------------------------------------------------------------------
    function [GlobalFrames,PartyMembers]=ChooseClusters(span,Npop,Nclusters,agent,GlobalFrames)
%---------------------------------------------------------------------------
%---------------------------------------------------------------------------
% defines the global frames for quantum model
% Input
% span=size of the circle in which the angels are projected
% Npop=# of agents
% Nclusters=how many global frames we want
% agent=structure with agents personalities
% GlobalFrames=current global frames
% Output
% GlobalFrames=new global frames

[RawFrames,~,datalabels]=kmeans_Angles(mod([agent.frame]',span),Nclusters,GlobalFrames');
RawFrames=RawFrames';
while any(isnan(RawFrames))
    whichOne=find(isnan(RawFrames));
    RawFrames(whichOne)=rand*span;
    [RawFrames,~,datalabels]=kmeans_Angles(mod([agent.frame]',span),Nclusters,GlobalFrames');
    RawFrames=RawFrames';
end  
PartyMembers=hist(datalabels,[1:Nclusters]);

GlobalFrames=sort(mod(RawFrames,span));%used to give the right color to the global frames
%---------------------------------------------------------------------------
%---------------------------------------------------------------------------
        function [Update_Global,Update_Local,action,ent,chosenFrame]=DecisionMaking(t,AgentFrame,AgentState,Comformity,Consistency,DecisionModel,probability,GlobalFrames);
%---------------------------------------------------------------------------
%---------------------------------------------------------------------------
% stocastically decides what action to take according to personality
% Input
% t=iteration #
% AgentFrame= agent current frame
% AgentState= agent current state
% Comformity=agent conformity
% Consistency=agent consistency
% DecisionModel=which decision model to use
% probability=random number for agent
% GlobalFrames = obvious
% Output
% Update_Global=how much to move towards global frame
% Update_Local=how much to move towards local frame
% action=which action to take
% ent=agent entropy
% chosenFrame=which global frame is closer to the agent
%---------------------------------------
%         Apply the decision making model
%---------------------------------------
% probability to choose according to the local frame
Nclusters=length(GlobalFrames);
Ent=zeros(1,Nclusters);prob_G=zeros(Nclusters,2);
angleLocal=AngleDistance(AgentFrame,AgentState);
prob_local=[cosd(angleLocal)^2 sind(angleLocal)^2 ]; 
local_ent=Entropy(prob_local);
%---------------------------------------
%         angle between global frame and state 
%         choose the global frame closer to the agent
%---------------------------------------
for n=1:Nclusters
    angle=AngleDistance(GlobalFrames(n),AgentState);
    DistAng(n)=min(angle,abs(90-angle));
    prob_G(n,:)=[cosd(angle)^2 sind(angle)^2]; 
    Ent(n)=Entropy(prob_G(n,:));
end
[~,chosenFrame]=min(Ent);
prob_global=prob_G(chosenFrame,:);
global_ent=Ent(chosenFrame);
switch DecisionModel
    case 'Weighted';
        P_Global=Comformity;P_Local=Consistency;%probability of choosing according to local vs global frame
        Update_Global=Comformity;Update_Local=Consistency;%once an action is taken, movement towards the local vs global frame
        %   weight and normalise the probabities
        P_sum=P_Global+P_Local;P_Global=P_Global/P_sum;P_Local=P_Local/P_sum;
        %   calculate entropy (Eq 13 in paper - notice the normalisation, not sure about this)
        ent=dot([local_ent global_ent],[P_Local P_Global]);
        prob=[prob_local*P_Local prob_global*P_Global];
        %choose an action (this part is a bit cryptic.. don't worry about it)
        prob=cumsum(prob)/sum(prob);bin=probability>prob;action=sum(bin)+1;
    case 'Ideology2'
        P_Global=Comformity;P_Local=1-P_Global;%probability of choosing according to local vs global frame
        Update_Global=Consistency;Update_Local=Consistency;%once an action is taken, movement towards the local vs global frame
%         convert the distance towards each global frame into a weight
        K=45;DistAng=K-DistAng;DistAng=DistAng/K;DistAng=DistAng/sum(DistAng);
        %   weight and normalise the probabities
        P_sum=P_Global+P_Local;P_Global=P_Global/P_sum;P_Local=P_Local/P_sum;
%         weight the distance to the global frames according to the conformity weight
        DistAng=P_Global*DistAng;
        %   calculate entropy (Eq 13 in paper - notice the normalisation, not sure about this)
        ent=dot([local_ent global_ent],[P_Local P_Global]);
%         apply weights to the probability for each global frame
        Prob_W=[DistAng' DistAng'];
        prob_G=prob_G.*Prob_W;
        prob_G=reshape(prob_G',1,numel(prob_G));
%         build the full probability
        prob=[prob_local*P_Local prob_G];
        %choose an action (this part is a bit cryptic.. don't worry about it)
        prob=cumsum(prob)/sum(prob);bin=probability>prob;action=sum(bin)+1;
%         ensure the action is bet 1 and 4 and choose the right global frame
        Act=action;
        if Act>2
            chosenFrame=floor((Act-1)/2+1)-1;
            action=3;
            if ~mod(Act,2),action=4;end
        end
    otherwise
        error('DecisionModel missing')
end
%---------------------------------------------------------------------------
%---------------------------------------------------------------------------
    function [AgentState,AgentFrame]=Act(action,AgentState,AgentFrame,GlobalFrame,Update_Local,Update_Global,span)
%---------------------------------------------------------------------------
%---------------------------------------------------------------------------
% moves frame and state according to decision taken
% Input
% action=what action has been taken
% AgentState= agent current state
% AgentFrame= agent current frame
% GlobalFrame=chosen global frame
% Update_Global=how much to move towards global frame
% Update_Local=how much to move towards local frame
% Output
% AgentState= agent new state
% AgentFrame= agent new frame
switch action
    case 1 % agent chose action 1 on the local frame
        [~,dir]=AngleDistance(AgentFrame,AgentState,1);%a1=mod(agent.state(i)+dir,span);fprintf('dir=%g, a1=%g, frame=%g \n',dir,a1,agent.frame(i));%pause
        AgentState=mod(AgentState+dir*Update_Local,span);
    case 2 % agent chose action 0 on the local frame     
        [~,dir]=AngleDistance(AgentFrame,AgentState,0);
        AgentState=mod(AgentState+dir*Update_Local,span);
    case 3 % agent chose action 1 on the global frame BOTH FRAME AND STATE CHANGE
        [~,dir]=AngleDistance(GlobalFrame,AgentState,1);
        AgentState=mod(AgentState+dir*Update_Global,span);
        [~,dir]=AngleDistance(GlobalFrame,AgentFrame,1);
        AgentFrame=mod(AgentFrame+dir*Update_Global,span);             
    case 4 % agent chose action 0 on the global frame BOTH FRAME AND STATE CHANGE
        [~,dir]=AngleDistance(GlobalFrame,AgentState,0);
        AgentState=mod(AgentState+dir*Update_Global,span);
        [~,dir]=AngleDistance(GlobalFrame,AgentFrame,0);
        AgentFrame=mod(AgentFrame+dir*Update_Global,span);
    otherwise
    error('Action missing')
end
% -----------------------------------------
% -----------------------------------------
        function [agent,GlobalFrames,randomNumbers,seed]=RandomStuff(Npop,Niter,seed,RandomPersonality,Nclusters,span)
% ------------------------------------------
% -----------------------------------------
% Input
% Npop=number of agents
% Niter=number of iterations
% seed= random seed (0=rendom,otherwise use the random seed to replicate a run)
% RandomPersonality = (0,1) (1= random conformity and consistency
%                           (0= conformity and consistency =.5 for each agent) 
% Nclusters= number of global frames
% span=size of the circle in which the angels are projected
if seed==0,seed=sum(100*clock);end;rand('state',seed);fprintf('seed=%g\n',seed);
randomNumbers=rand(Niter,Npop);%storing random numbers
agent=struct('frame',mod([rand(1,Npop)*span],span),'state',mod([rand(1,Npop)*span],span),'comformity',[rand(1,Npop)],'consistency',[rand(1,Npop)]);
% agent structure (comformity=desire to conform to others, consistency=probability of taking the same action if asked twice=need for structure 
if ~RandomPersonality
    agent.comformity=ones(1,Npop)*.5;agent.consistency=ones(1,Npop)*.5;
end
%initialising GlobalFrames for first clustering iteration 
GlobalFrames=rand(1,Nclusters)*span;GlobalFrames=mod(GlobalFrames,span);
% -----------------------------------------
% -----------------------------------------
        function WriteFrames(GlobalFrames,PartyMembers,t)
% ------------------------------------------
% -----------------------------------------
GF=[GlobalFrames GlobalFrames(1)];
fprintf('t=%g, Global Frames=',t);
fprintf('%g ',round(GlobalFrames));
fprintf('distance Bet Frames=');
fprintf('%g ',round(abs(diff(GF))));
fprintf('| Membership=');
fprintf('%g ',PartyMembers/sum(PartyMembers));
fprintf('\n')

% -----------------------------------------
% -----------------------------------------
        function [c,costfunctionvalue, datalabels,inter] = kmeans_Angles(data,k,varargin);
% ------------------------------------------
% -----------------------------------------
% Kmeans algorhthm modified by Fabio Boschetti to account for difference between angle
% originally from
% kmeans.m : K-means clustering algorithm
% Copyright (c) 2002 - 2003  
% Jussi Tohka
% *****************************************************************
% [c,costfunctionvalue, datalabels] = kmeans(data,k,c_init,max_iter)
% Input:
% data is the n x m matrix, where n is the number of data points
% and m is their dimensionality. 
% k is the number of clusters
% c_init is the initializations for cluster centres. This must be a 
% k x m matrix. (Optional, can be generated randomly). 
% max_iter is the maximum number of iterations of the algorithm
% (Default 50).
% Output:
% c is the  k x m matrix of final cluster centres.
% costfunctionvalue is the value of cost function after each
% iteration.
% datalabels is a n x 1 vector of labeling of data. 
   [n m] = size(data);
   if n < m
     fprintf(1,'Error: The number of datapoints must be greater than \n');
     fprintf(1,'their dimension. \n');
     return;
   end
   if length(varargin) > 0
     c_init = varargin{1};
   else
     % First, select k random numbers from 1 to n WITHOUT repetition
     randomindex = randperm(n);
     randomindex=randomindex(1:k);
     c_init = data(randomindex,:);
   end
   if size(c_init) ~= [k m];
     fprintf(1,'Error: The size of c_init is incorrect.');
   end
   if length(varargin) > 1
     max_iter = varargin{2};
   else
     max_iter = 50;
   end
   % Start the algorithm
   iter = 0;
   changes = 1;
   distances = zeros(n,k);
   costfunctionvalue = zeros(max_iter + 1,1);
   c = c_init;
   datalabels = zeros(n,1);
   while iter < max_iter & changes
     iter = iter + 1;
%      fprintf(1,'#');
     old_datalabels = datalabels;
     % Compute the distances between cluster centres and datapoints
     for i = 1:k
         dist(:,i) = sum(AngleDistance(data,c(i,:)),2);
     end
     % Label data points based on the nearest cluster centre
     [tmp,datalabels] = min(dist,[],2);
     % compute the cost function value
     costfunctionvalue(iter) = sum(tmp);
     % calculate the new cluster centres 
     for i = 1:k
         newdata=data(find(datalabels == i),:);
       c(i,:) = AngleMean(newdata);
     end
     % study whether the labels have changed
     changes = sum(old_datalabels ~= datalabels);
     inter(iter).datalabels = datalabels;
     inter(iter).c = c;
   end
   for i = 1:k
     dist(:,i) = sum(AngleDistance(data,c(i,:)),2);
   end
   [tmp,datalabels] = min(dist,[],2);
   % compute the cost function value
   costfunctionvalue(iter + 1) = sum(tmp);

% -----------------------------------------
% -----------------------------------------
        function [angle,dir]=AngleDistance(angle1,angle2,vote);
% ------------------------------------------
% -----------------------------------------
angle=abs(mod(angle1,180)-mod(angle2,180));
angle=min(angle,180-angle);
%-------------------
if nargin>2
    Err=1e-8;
    if vote,
        dir=angle;
        a2=angle2+dir;
        a=abs(mod(angle1,180)-mod(a2,180));
        a=min(a,180-a);
        if abs(a)>Err,dir=-dir;end
        a2=angle2+dir;
        a1=abs(mod(angle1,180)-mod(a2,180));
        a1=min(a1,180-a1);
        if abs(a1)>Err
            fprintf('a=%g, a1=%g,angle1=%g, angle2=%g, dir=%g, vote=%g, angle=%g \n',a,a1,angle1,angle2,dir,vote,angle)
            pause
        end
    else
        dir=90-angle;
        a2=angle2+dir;
        a=abs(mod(angle1+90,180)-mod(a2,180));
        a=min(a,180-a);
        if abs(a)>Err,dir=-dir;end
        a2=angle2+dir;
        a1=abs(mod(angle1+90,180)-mod(a2,180));
        a1=min(a1,180-a1);
        if abs(a1)>Err
            fprintf('Err=%g, a=%g,a1=%g, angle1=%g, angle2=%g, dir=%g, vote=%g, angle=%g\n',Err,a,a1,angle1,angle2,dir,vote,angle)
            pause
        end
    end
end
% -----------------------------------------
% -----------------------------------------
        function [idx]=find_idx(centres,distance)
% ------------------------------------------
% -----------------------------------------
% given a set of centres and a distance matrix, find which centre each point belongs to
% Input
% centres = cluster centres
% distance=distance matrix betwenn points
% Output
% idx=labels, from clustering algorithm
dist=distance(:,centres);
[C,I] = min(dist,[],2);
idx=centres(I);
% -----------------------------------------
% -----------------------------------------
        function [angleOut]=AngleMean(angles)
% ------------------------------------------
% -----------------------------------------
ALLAngles=[0:1:179];
dmin=1e5;angleOut=0;
for i=1:length(ALLAngles)
    d1=sum(AngleDistance(angles,ALLAngles(i)).^2);
    if d1<dmin;dmin=d1;angleOut=ALLAngles(i);end
end

% -----------------------------------------
% -----------------------------------------
        function PlotPolar2(t,Npop,Nclusters,agent,GlobalFrames,FontSize,ImageFrames,ImageName,hFig)
% ------------------------------------------
% -----------------------------------------

clf;
factAgent=1;factFrame=.75;factGlobal=.87;maxFact=max([factAgent factFrame factGlobal])*1.1;
MS=35;
ColClust={'.r','.b','.k','.g','.c','.m','.y'};
state=mod(agent.state,180);
frame=mod(agent.frame,180);
Global=mod(GlobalFrames,180);
%     agent frames

for i=1:Npop
        X=[0 cosd(state(i))]*factFrame;Y=[0 sind(state(i))]*factFrame;
        plot(X,Y,'k');hold on           
end
%     agents
for i=1:Npop
        X=[cosd(frame(i))]*factAgent;Y=[sind(frame(i))]*factAgent;
        plot(X,Y,'.k');hold on
end
%   clusters        
for c=1:Nclusters
        X=cosd(Global(c))*factGlobal;
        Y=sind(Global(c))*factGlobal;
        Col=ColClust{c};%select the right color according to the order of the frames
        plot(X,Y,Col,'MarkerSize',MS);hold on
end
Title=strcat('Frames & States: t=',num2str(t));
title(Title,'fontsize',FontSize,'fontweight','bold')
axis equal
ylim([-0.1 maxFact]);xlim([-maxFact maxFact]);
set(gca,'Fontweight','b');%for ticks
%     axis([-maxFact maxFact 0 maxFact]);
if ismember(t,ImageFrames)
    filename=strcat(ImageName,num2str(t),'.png');
    fprintf('Printing %s \n',filename)
    print(hFig,'-dpng',filename)
end

pause(.01)
% -----------------------------------------
% -----------------------------------------
function H = Entropy(S)
% -----------------------------------------
% -----------------------------------------
% H = entropy(S)
% Function returns 0th order entropy of a source.
% S is probability or count of each symbol
% S should be a vector of non-negative numbers.

% Ver. 1.0  09.10.97  Karl Skretting
% Ver. 1.1  25.12.98  KS, Signal Processing Project 1998, english version

if nargin<1
   error('entropy: see help.')
end

N=sum(sum(S));		% if S is probability this is 1
Snz=nonzeros(S);
H=log2(N)-sum(Snz.*log2(Snz))/N;
return
